package org.vishia.odt.readOdt;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;
import java.util.TreeMap;

import org.vishia.msgDispatch.LogMessage;
import org.vishia.util.Arguments;
import org.vishia.util.Debugutil;
import org.vishia.util.FileFunctions;
import org.vishia.util.StringFunctions;

abstract class TranslateOdtCommon {

  
  
  protected static abstract class CommonArgs extends Arguments {
    
    protected static class DirName {
      File dir;
      String name;
    }
    
    String sCurrDir;
    
    /**The input file to read, either odt or ZmL. */
    File dirIn, fIn;

    /**If not null then write some report info. */
    File fReport;
    

    /**If not null then write the log output beside stdout, stderr in this file. Argument -log:path/to. */
    OutputStream fLog;
    

    /**This should be present the component's directory in the 'src/component'. 
     * <ul>
     * <li>On ReadOdt argument '-odt:path/to/docu++file.odt' it is set with the left part of the odt file name before '++'.
     * <li>or it is set to null if the file does not contain '++'.
     * <li>On WriteOdt it is set with the directory name of the '--dirCmpn=path/to/component', usual '--dirCmpn=../..'
     * <li>or remains null if this argument is not given. 
     * </ul>
     * This variable is used:
     * <ul>
     * <li>On ReadOdt to determine paths in '-ZmL:src/default++/*.ZmL' replacing the '++'
     * <li>On WriteOdt to determine a first part of the odt output file <&sDirCmpn>++<&sNameDoc>.odt
     * </ul>
     * null or the first part of the name of {@link #fIn} if this name contains "++" in mid.
     * The ++ is adequate a separator but admissible for file names. */
    protected String sDirSrcCmpn;
    
    /**This is either the whole name of {@link #fIn} or the right part after a "++" in the name, without extension.
     * It is used to build the name of the ZmL file as <&sNameDoc>.ZmL */
    String sNameDoc;
    
    String sExtOut;
    
    /**Directory (relative to current directory where the *.odt invokes this conversion)
     * where the Zml and also label files are stored.
     */
    File dirZml;
    
    /**Used extension for the ZmL file.
     */
    String sExtZmL;
    
    /**If set then create backup files for the current file.ZmL before writing the new one. 
     * 
     */
    File dirOutBack;
    
 
    /**If given as true by "-backall:dir", then rename all existing backup files with a number (as "file.99.vml") to preserve all versions.
     * Set to false by "-back:dir"
     */
    boolean bBackAll;
    
    /**This is either name and ext from the input markup file on read ZmL or same for output file on write ZmL.*/
    //String sNameMarkup, sExtMarkup;
    

    /**The directory for included files to read for ({@link WriteOdt}) or for write ({@link ReadOdt})
     * due to the command line argument '-fincl:path/*.ZmL' */
    //File dirInclZml;
    
    /**The name parts and the extension for included files to read for ({@link WriteOdt}) or for write ({@link ReadOdt})
     * due to the command line argument '-fincl:path/nameStart*nameEnd.Ext' */
    //String sNameStartInclZml, sNameEndInclZml, sExtInclZml;
    
    
    /**Directory of the label file.*/
    public File dirLabel;
    
    /** "*" if * is written, read all files. */
    public String sNameLabel;
    

    /**Contains a relative (or possible absolute) or also a URL path where foreign linked bookmark files are placed. 
     * It has the form "path/*.ext" where ".ext" may be ".html" to link to html files.
     */
    String sRefBesideHtml, sRefBesidePdf;
    

    /**Contains the extension for reference links beside the own document from -labels. 
     * It has the form "path/*.ext" where ".ext" may be ".html" to link to html files.
     */
    String sExtRefBesideHtml = ".html", sExtRefBesidePdf = ".pdf";
    

    /**Contains the www URL path where the files with references to bookmarks to documents beside are placed
     * additional to the given #-sRefBesidePdf
     * It has the form "https://myPage.org/path/*.ext" where ".ext" may be ".html" to link to html files.
     */
    String sRefBesideInWWWHtml, sRefBesideInWWWPdf;
    

    File dirDbg;
    
    
    
    public void setCurrDir() {
      String  sCurrDir1 = System.getProperty("user.dir");
      sCurrDir1 = FileFunctions.getCanonicalPath(new File(sCurrDir1));
      this.sCurrDir = sCurrDir1; 
    }
    
    
    Argument[] argListCommon = {
        new Argument("-dirDbg", ":dirWrTest    directory for debug output", new SetArgument(){ 
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            CommonArgs.this.dirDbg = getDirRepaceSrcCmpn(val);  
            FileFunctions.mkDir(CommonArgs.this.dirDbg);
            return CommonArgs.this.dirDbg.exists();
          }})
        , new Argument("-dirCmpn", ":../.. directory from current for the src/Component replacing ++ in odt:++NAME", new SetArgument(){ 
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            File fileCmpn = new File(val);
            boolean bOk = fileCmpn.exists();
            if(bOk) {
              String sFileCmpnAbs = FileFunctions.normalizePath(fileCmpn).toString();
              int posLastSlash = sFileCmpnAbs.lastIndexOf('/');
              CommonArgs.this.sDirSrcCmpn = sFileCmpnAbs.substring(posLastSlash+1);
            }
            return bOk;
          }})

//        , new Argument("-fincl", ":path/start*end.ZmL  dir / file mask for included ZmL files", new SetArgument(){ 
//          @Override public boolean setArgument(String val) throws FileNotFoundException { 
//            //int posWildcard = val.indexOf('*');
//            int posDirEnd = val.lastIndexOf('\\')+1;
//            int posDirEnd2 = val.lastIndexOf('/')+1;
//            if(posDirEnd2 > posDirEnd) { posDirEnd = posDirEnd2; }
//            int posDot = val.indexOf('.', posDirEnd);
//            int posWildcard = val.indexOf('*', posDirEnd);
//            if(posDirEnd ==0 || posWildcard <0) {    //--------------------------- only director by one name is given
//              CommonArgs.this.sNameStartInclZml = ""; CommonArgs.this.sNameEndInclZml = ""; CommonArgs.this.sExtInclZml = ".ZmL";
//              posDirEnd = val.length();   // whole String is the dir
//            } else if(posDot <0){
//              CommonArgs.this.sNameStartInclZml = val.substring(posDirEnd, posWildcard);  // often ""
//              CommonArgs.this.sNameEndInclZml = val.substring(posWildcard+1);  // often ""
//              CommonArgs.this.sExtInclZml = ".ZmL";
//            } else {
//              CommonArgs.this.sNameStartInclZml = val.substring(posDirEnd, posWildcard);  // often ""
//              CommonArgs.this.sNameEndInclZml = val.substring(posWildcard+1, posDot);  // often ""
//              CommonArgs.this.sExtInclZml = val.substring(posDot);
//            }
//            CommonArgs.this.dirInclZml = new File(posDirEnd == 0 ? "." : val.substring(0, posDirEnd)).getAbsoluteFile();
//            return CommonArgs.this.dirInclZml.exists();
//          }})
      , new Argument("-labels", ":path/*.label.txt  file contains labels", new SetArgument(){ 
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            int posEndDir = val.lastIndexOf('/');
            if(posEndDir <0) {
              CommonArgs.this.dirLabel = CommonArgs.this.dirZml;
            } else {
              CommonArgs.this.dirLabel = getDirRepaceSrcCmpn(val.substring(0, posEndDir));
            }
            CommonArgs.this.sNameLabel = val.substring(posEndDir+1);  // not: it is from 0 if no dir is given
            return CommonArgs.this.dirLabel.exists();
          }})
      , new Argument("-report", ":path/default++/file.txt  file for report", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          DirName dirName = getDirNameReplaceSrcCmpn(val);
          CommonArgs.this.fReport = new File(dirName.dir, dirName.name);
          FileFunctions.mkDir(CommonArgs.this.fReport.getParentFile());
          return CommonArgs.this.fReport.getParentFile().exists();
        }})
      , new Argument("-log", ":path/default++/file.log  file for report", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          DirName dirName = getDirNameReplaceSrcCmpn(val);
          File fLog = new File(dirName.dir, dirName.name);
          FileFunctions.mkDir(fLog.getParentFile());
          CommonArgs.this.fLog = new FileOutputStream(fLog);
          return CommonArgs.this.fLog !=null;
        }})
      , new Argument("-back", ":path/to  dir for backup vml files", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          CommonArgs.this.dirOutBack = getDirRepaceSrcCmpn(val);  
          CommonArgs.this.bBackAll = false;
          FileFunctions.mkDir(CommonArgs.this.dirOutBack);
          return CommonArgs.this.dirOutBack.exists();
        }})
      , new Argument("-relhtml", ":../html/*.html relative path for extern beside bookmark references", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          if(val.equals("?")) {
            CommonArgs.this.sRefBesideHtml = null;
          } else {
            int posWildcard = val.indexOf('*');
            if(posWildcard <0) {
              if(val.endsWith("/")) {
                CommonArgs.this.sRefBesideHtml = val;
              } else {
                CommonArgs.this.sRefBesideHtml = val + "/";
              }
            } else {
              CommonArgs.this.sRefBesideHtml = val.substring(0, posWildcard);
              CommonArgs.this.sExtRefBesideHtml = val.substring(posWildcard +1);
            }
          }
          return true;
        }})
      , new Argument("-relpdf", ":../pdf/*.pdf relative path for extern beside bookmark references", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          if(val.equals("?")) {
            CommonArgs.this.sRefBesidePdf = null;
          } else {
            int posWildcard = val.indexOf('*');            // Note: the "../" is necessary from inside content.xml as relative path
            if(posWildcard <0) {
              if(val.endsWith("/")) {
                CommonArgs.this.sRefBesidePdf = val;
              } else {
                CommonArgs.this.sRefBesidePdf = val + "/";
              }
            } else {
              CommonArgs.this.sRefBesidePdf = val.substring(0, posWildcard);
              CommonArgs.this.sExtRefBesidePdf = val.substring(posWildcard +1);
            }
          }
          return true;
        }})
      , new Argument("-arelhtml", ":D:/path/*.html OR https://www.xyz.org/path/*.html path for extern beside bookmark references", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          int posWildcard = val.indexOf('*');
          if(posWildcard <0) {
            if(val.endsWith("/")) {
              CommonArgs.this.sRefBesideHtml = val;
            } else {
              CommonArgs.this.sRefBesideHtml = val + "/";
            }
          } else {
            CommonArgs.this.sRefBesideHtml = val.substring(0, posWildcard);
            CommonArgs.this.sExtRefBesideHtml = val.substring(posWildcard +1);
          }
          return true;
        }})
      , new Argument("-arelpdf", ":D:/path/*.pdf OR https://www.xyz.org/path/[*.pdf] path for extern beside bookmark references", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          int posWildcard = val.indexOf('*');
          if(posWildcard <0) {
            if(val.endsWith("/")) {
              CommonArgs.this.sRefBesidePdf = val;
            } else {
              CommonArgs.this.sRefBesidePdf = val + "/";
            }
          } else {
            CommonArgs.this.sRefBesidePdf = val.substring(0, posWildcard);
            CommonArgs.this.sExtRefBesidePdf = val.substring(posWildcard +1);
          }
          return true;
        }})
      , new Argument("-wrelhtml", "https://www.xyz.org/path/*.pdf path for extern beside bookmark references", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          if(val.equals("?")) {
            CommonArgs.this.sRefBesideInWWWHtml = null;
          } else {
            int posWildcard = val.indexOf('*');
            if(posWildcard <0) {
              if(val.endsWith("/")) {
                CommonArgs.this.sRefBesideInWWWHtml = val;
              } else {
                CommonArgs.this.sRefBesideInWWWHtml = val + "/";
              }
            } else {
              CommonArgs.this.sRefBesideInWWWHtml = val.substring(0, posWildcard);
              CommonArgs.this.sExtRefBesideHtml = val.substring(posWildcard +1);
            }
          }
          return true;
        }})
      , new Argument("-wrelpdf", "https://www.xyz.org/path/*.pdf path for extern beside bookmark references", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          if(val.equals("?")) {
            CommonArgs.this.sRefBesideInWWWPdf = null;
          } else {
            int posWildcard = val.indexOf('*');
            if(posWildcard <0) {
              if(val.endsWith("/")) {
                CommonArgs.this.sRefBesideInWWWPdf = val;
              } else {
                CommonArgs.this.sRefBesideInWWWPdf = val + "/";
              }
            } else {
              CommonArgs.this.sRefBesideInWWWPdf = val.substring(0, posWildcard);
              CommonArgs.this.sExtRefBesidePdf = val.substring(posWildcard +1);
            }
          }
          return true;
        }})
      , new Argument("-alinkhtml", ":D:/path/*.html OR https://www.xyz.org/path/*.html deprecated: path for extern bookmark references", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          int posWildcard = val.indexOf('*');
          if(posWildcard <0) {
            if(val.endsWith("/")) {
              CommonArgs.this.sRefBesideHtml = val;
            } else {
              CommonArgs.this.sRefBesideHtml = val + "/";
            }
          } else {
            CommonArgs.this.sRefBesideHtml = val.substring(0, posWildcard);
            CommonArgs.this.sExtRefBesideHtml = val.substring(posWildcard +1);
          }
          return true;
        }})
      , new Argument("-alinkpdf", ":D:/path/*.pdf OR https://www.xyz.org/path/[*.pdf] deprecated: path for extern bookmark references", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          int posWildcard = val.indexOf('*');
          if(posWildcard <0) {
            if(val.endsWith("/")) {
              CommonArgs.this.sRefBesidePdf = val;
            } else {
              CommonArgs.this.sRefBesidePdf = val + "/";
            }
          } else {
            CommonArgs.this.sRefBesidePdf = val.substring(0, posWildcard);
            CommonArgs.this.sExtRefBesidePdf = val.substring(posWildcard +1);
          }
          return true;
        }})
      , new Argument("-rlinkhtml", ":../html/*.html OR https://www.xyz.org/path/*.html path for foreign bookmark references", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          int posWildcard = val.indexOf('*');            // Note: the "../" is necessary from inside content.xml as relative path
          if(posWildcard <0) {
            if(val.endsWith("/")) {
              CommonArgs.this.sRefBesideHtml = "../" + val;
            } else {
              CommonArgs.this.sRefBesideHtml = "../" + val + "/";
            }
          } else {
            CommonArgs.this.sRefBesideHtml = "../" + val.substring(0, posWildcard);
            CommonArgs.this.sExtRefBesideHtml = val.substring(posWildcard +1);
          }
          return true;
        }})
      , new Argument("-rlinkpdf", ":../pdf/*.pdf OR https://www.xyz.org/path/*.pdf path for foreign bookmark references", new SetArgument(){ 
        @Override public boolean setArgument(String val) throws FileNotFoundException { 
          int posWildcard = val.indexOf('*');            // Note: the "../" is necessary from inside content.xml as relative path
          if(posWildcard <0) {
            if(val.endsWith("/")) {
              CommonArgs.this.sRefBesidePdf = "../" + val;
            } else {
              CommonArgs.this.sRefBesidePdf = "../" + val + "/";
            }
          } else {
            CommonArgs.this.sRefBesidePdf = "../" + val.substring(0, posWildcard);
            CommonArgs.this.sExtRefBesidePdf = val.substring(posWildcard +1);
          }
          return true;
        }})
  
    };

    
    /**Combines the given 'val' with {@link #sDirSrcCmpn} to a directory and extract maybe the given extension.
     * This operation is used for several argument preparation.
     * <ul>
     * <li>If '++/' is found in args, and {@link #sDirSrcCmpn} is set (not null), 
     *   then the directory part with '++' on end is replaced by {@link #sDirSrcCmpn}.
     * <li>If '++/' is found, but {@link #sDirSrcCmpn} null, 
     *   (because the input file name has not contained a '++' in the name,
     *   or argument '-dirCmpn' was not given) 
     *   then '++' is removed in the path, it means the directory name before '++' is used.
     *   This should be the default component path.
     * <li>If '++/' is not found, then the directory is built as given.
     * </ul>
     * Only on 'bFile' = true: If '*' is containd in val, then:
     * <ul>
     * <li> '/*' should be written, as last val path element. 
     * <li> then {@link DirName#name} is set with the part after *
     * <li> If '*' is not found, then {@link DirName#name} is set with the last path entry as file name. 
     * </ul>
     * @param val argument value can contain 'path/default++/to/*name.ext'
     * @param bFile true then extract an file name or extension, false: build the directory with val
     * @return
     */
    protected DirName getDirNameReplaceSrcCmpn(String val) {
      DirName dirExt = new DirName(); // only used for return.
      int posEndDir;
      posEndDir = val.lastIndexOf('/');
      int posWildcard = val.indexOf("*");
      int posWildcard2 = val.indexOf("++*");
      if(posWildcard2 >=0) { posWildcard = posWildcard2; } 
      if(posWildcard >= 0) {
        if(posEndDir != posWildcard-1) {
          throw new IllegalArgumentException("Wildcard can only be used replacing the name as 'path/to/*.ext', found: '" + val + "'");
        } else if(posWildcard2 >0) {
          dirExt.name = this.sDirSrcCmpn + "++" +  this.sNameDoc + val.substring(posWildcard2 +3);
        } else {  
          dirExt.name = this.sNameDoc + val.substring(posWildcard +1);
        }
      } else {
        dirExt.name = val.substring(posEndDir+1);    // the 'name.ext' after last /
      }
      dirExt.dir = getDirRepaceSrcCmpn(val.substring(0, posEndDir));
      return dirExt;
    }

    
    protected File getDirRepaceSrcCmpn(String sDirArg) {
      String sDir;
      int posSepDir = sDirArg.indexOf("++");
      if(posSepDir >0) {
        if(this.sDirSrcCmpn ==null) {
          sDir = sDirArg.substring(0, posSepDir) + sDirArg.substring(posSepDir+2);  // without the '...++..., the default dir
        } else {
          int posDirBeforeSep = sDirArg.lastIndexOf("/", posSepDir);               // use the first part of doc name for sDirZml
          sDir = sDirArg.substring(0, posDirBeforeSep+1) + this.sDirSrcCmpn + sDirArg.substring(posSepDir +2);
        }
      } else {
        sDir = sDirArg;  // use as given
      }
      return new File(sDir);  
    }
    
    
    
    
  }    

  
  /**Central possibility to write messages. Can be in file, on screen. */
  final public LogMessage log;


  
  
  final GetJavadocLabel javadocLabel;
  
  protected TranslateOdtCommon (LogMessage log) {
    this.log = log;
    this.javadocLabel = new GetJavadocLabel(log);
  }
  
  
  
  /**Replace the sTarget with the correct javadoc-operation target if it is an operation written with '(...)',
   * or also search the given javadoc-operation and check whether it is unique.
   * This operation should be only called either if the javadoc-operation is written with "(...)" (ZmL to odt)
   * or if the javadoc-operation is given and should be tested whether it is unique (odt to ZmL)
   * <ul>
   * <li>With 'sHtml', the proper entry in {@link GetJavadocLabel#idxHtmlFilesAnchors} will be gotten,
   *   or the html file will be parsed now, and entered in this index.
   * <li>The 'name' (java-identifier) from sTarget is extracted, sTarget should start with a name either from an operation
   *   or also it is the name of a field. 
   * <li>with the gotten 'idxOperations' and the 'name' the proper anchor (inner link target) is searched.
   *   if it is found, this is the proper target for the link which is returned.
   * <li>If it is not found, but it is not unique, the '?' on end is the signal to preserve the link while converting odt to ZmL
   * <li>If it is not found, the sTarget is returned to use unchanged.
   * </ul>
   * @param sHtml
   * @param sTarget
   * @param dirBase
   * @return sTarget itself or the correct operation target.
   */
  protected String searchForReplacingLinkOperationLabel( String sHtml, String sTarget, File dirBase) { //String sRef, int posFileEnd, File dirBase) {
    final String retTarget;
    //if(sHtml.contains("XmlSequWriter")) Debugutil.stopp(); 
    Map<String, String> idxOperations = this.javadocLabel.idxHtmlFilesAnchors.get(sHtml);
    try {
      if(idxOperations == null) {
        File fHtml = new File(dirBase, sHtml);
        if(fHtml.exists()) {
          idxOperations = this.javadocLabel.createHtmlAnchorOperations(sHtml, dirBase);
        } else {
          this.log.writeError("Warning: file for searchForReplacingLinkOperationLabel in html files not found: '%s' ", fHtml.getAbsolutePath());
        }
      }
    } catch(IOException exc) {
      this.log.writeError("ERROR read html file '%s'", sHtml);
    }
    if(idxOperations !=null) {
      int posNameEnd2 = StringFunctions.indexAfterIdentifier(sTarget, 0, -1, null);
      String name;
      if(StringFunctions.equals(sTarget, posNameEnd2, posNameEnd2+2, "()")) {
        name = sTarget.substring(0, posNameEnd2+2); // Hint: posName is either -1 or position of '.' or '#'
      } else {
        name = sTarget.substring(0, posNameEnd2); // Hint: posName is either -1 or position of '.' or '#'
      }
      //int posName = sRef.lastIndexOf('#', posNameEnd);
      //if(posName <0) { posName = sRef.lastIndexOf('.', posNameEnd); }
      String sOperAnchor = idxOperations.get(name);
      if(sOperAnchor == null) {
        this.log.writeError("Warning: operation '%s' not found in: '%s' ", name, sHtml);
        retTarget = sTarget;
      } else if(sOperAnchor.endsWith("?")) {
        this.log.writeError("Warning: operation '%s' is not unique in: '%s' ", name, sHtml);
        retTarget = sTarget;
      } else {
        retTarget = sOperAnchor;
      }
    } else {
      retTarget = sTarget;
    }
    return retTarget;
  }
  
  
  
  /**Gets the name of the linked file inside a URL, without extension. 
   * @param sURL for example "../name.ext#Label"
   * @return for example "name"
   */
  public String getLinkNameFromURL(String sURL) {
    int posNameEnd = sURL.indexOf('#');
    if(posNameEnd < 0) { posNameEnd = sURL.length(); }
    int posName = sURL.lastIndexOf('/') +1;  // 0 if no /
    int posDot = sURL.lastIndexOf('.');
    if(posDot < posName) { posDot = posNameEnd; }
    return sURL.substring(posName, posDot);
  }
  
  
  
  /**Gets the file name of the linked file inside a URL, without extension. 
   * @param sURL for example "../name.ext#Label"
   * @return for example "name.ext"
   */
  public String getLinkFileFromURL(String sURL) {
    int posNameEnd = sURL.indexOf('#');
    if(posNameEnd < 0) { posNameEnd = sURL.length(); }
    int posName = sURL.lastIndexOf('/') +1;  // 0 if no /
    return sURL.substring(posName, posNameEnd);
  }
  
  
  
  /**Writes a backup file for the given output file to generate if the fOut exists.
   * This is also important to compare the generated ZmL files as also for data recovery on unexpected faults.
   * Depending from {@link CommonArgs#bBackAll} an existing backup file will be renamed before
   * with a current number testing all existing files.
   * @param fOut The output file, if exists make a copy
   * @param args arguments
   */
  protected void writeBackupFile (File fOut, File dirBack, String sBackExtArg, ReadOdt.CommonArgs args) {
    if(fOut.exists() && dirBack !=null) {
      String sNameBack = fOut.getName();
      int posExt = sNameBack.lastIndexOf('.');
      String sBackExt = sBackExtArg !=null ? sBackExtArg: sNameBack.substring(posExt);
      sNameBack = sNameBack.substring(0, posExt);
      if(FileFunctions.getCanonicalPath(dirBack).equals(FileFunctions.getCanonicalPath(fOut.getParentFile()))) {
        sNameBack += ".back";                              // if dirBack is the same as source dir
      }
      boolean bOkRename = true;
      if(args.bBackAll) {
        File fBack0 = new File(dirBack, sNameBack + sBackExt);
        if(fBack0.exists()) {
          int nr = 0;
          File fBack;
          do {
            String sFileBack = sNameBack + "." +  (++nr) + sBackExt;
            fBack = new File(dirBack, sFileBack);
          } while(fBack.exists());                     // search the first non exist file with nr.
          bOkRename = fBack0.renameTo(fBack);    // rename the last backup file with this number.
          if(bOkRename) { this.log.writef("\nsuccessfull backall: %s", fBack.getAbsolutePath()); }
          else { this.log.writef("\nERROR backall: %s", fBack.getAbsolutePath()); }
      } }
      //======>>>>                    =================== copy the vml back version.
      File fBack = new File(dirBack, sNameBack + sBackExt);
      try { 
        FileFunctions.copyFile(fOut, fBack);
        this.log.writef("\nsuccessfull back: %s", fBack.getAbsolutePath());
      } catch(IOException exc) {
        this.log.writef("\nERROR back: %s : %s", fBack.getAbsolutePath(), exc.getMessage());
      }
    }
  }


}
