package org.vishia.libOffc.cfgui;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.vishia.msgDispatch.LogMessage;
import org.vishia.util.Debugutil;
import org.vishia.util.FileFunctions;
import org.vishia.util.OutTextPreparer;

/**Operation class for a read XML file from LibreOffice Customize Keyboard Save
 * to convert it in a human readable form. 
 * <br><br>
 * This class is used in {@link WriteKeyAccl#execute()},
 * there only {@link #exec(XmlDataLOffcKeyAccl, org.vishia.libOffc.cfgui.WriteKeyAccl.CmdArgs, LogMessage)} 
 * is called with the already read XML data.
 * <ul>
 * <li>This class organizes calling a translation from uno commands to a human readable text.
 *   This is done with the class {@link ReadRegistryMain} which reads XML files in the LibreOffice tool files
 *   called in {@link #exec2(XmlDataLOffcKeyAccl)}.
 * <li>The key names in the XML file are translated, using an inner table {@link #idxKeyTranslateKeyNames}
 *   filled in {@link #initKeyTranslate()}.  
 * <li>Then it sorts the given data and builds the index of key acceleration {@link #idxKeyAccl}.
 * <li>The index is written to a text file using {@link #outKeys()}.
 *   For that the {@link OutTextPreparer} capability ('gTxt') is used.
 * <ul>
 */
public class WriteKeyAcclPrc {

  
  /**Data class to store properties of one accelerator key. */
  public static class AcclKey {
    
    /**The key identification as contained in the XML file */
    public final String keyRaw;
    
    /**Improved key name human readable. */
    public final String keyString1;
    
    /**Human readable key functionality after uno translation. */
    public final String sLabel_en;
    
    /**The original stored string for key functinality in the input XML file. */
    public final String sHref;

    /**ctor with all data.
     * @param keyRaw
     * @param keyString1
     * @param sLabel
     * @param sHref
     */
    public AcclKey( String keyRaw, String keyString1, String sLabel, String sHref ) {
      this.keyRaw = keyRaw;
      this.keyString1 = keyString1;
      this.sHref = sHref;
      this.sLabel_en = sLabel;
    }
    
    @Override public String toString() { return this.keyString1 + "=:" + this.sLabel_en; }
  }
  
  
  /**Command arguments from {@link WriteKeyAccl} main class. */
  final WriteKeyAccl.CmdArgs cmdArgs;
 
  /**Result accelerator key association to command. */
  final Map<String, AcclKey> idxKeyAccl = new TreeMap<>();
  
  /**Result accelerator key association to command in the order shift following no shift key. */
  final Map<String, AcclKey> idxKeyAcclShift = new TreeMap<>();
  
  
  /**Internal translation between key names in the XML file and human readable names. */
  final Map<String, String> idxKeyTranslateKeyNames = new TreeMap<>();

  
  /**Instance to translate the uno and other commands to human readable text.*/
  ReadRegistryMain reg = new ReadRegistryMain();
  
  /**The gTxt script which is used for output.
   * It is read from file, see {@link #cmdArgs} {@link WriteKeyAccl.CmdArgs#file_gTxt}
   * If this file is not given or faulty, then {@link #gTxtKeyOutputDefault} is used.
   */
  public final OutTextPreparer gTxtKeyOutput;

  /**The gTxt script {@link OutTextPreparer} as default for the output file as constant (final static).
   * It is completely parsed in its ctor (on class loading).
   * */
  public final static OutTextPreparer gTxtKeyOutputDefault = new org.vishia.util.OutTextPreparer("GTxtKeyOutput", WriteKeyAccl.class, "thiz, key, descr, uno"
      , "<:if:thiz.reg.idxUnoLabel.size()==0>PROBLEM uno translation not done because faulty -LOffcPrg:<&thiz.cmdArgs.dirLOffcPrg.getAbsolutePath()><:n><.if>"
        + "<:for:e:thiz.idxKeyAccl2>"
        + "<&e.keyString1><:tab:18>\"<&e.sLabel_en.replace('~', '')>\"<:tab:50>==><&e.sHref><:n>"
        + "<.for>"
      , null);

  /**Index of all read scripts, here only one expected.*/
  public final Map<String, OutTextPreparer> idxScript = new TreeMap<>();
  
  
  /**Ctor.
   * <ul>
   * <li>calls {@link #initKeyTranslate()}
   * <li> prepares the gTxt script given with {@link #cmdArgs} {@link WriteKeyAccl.CmdArgs#file_gTxt}
   *   If this file is not given or faulty, then {@link #gTxtKeyOutputDefault} is used.
   * </ul>  
   * @param log
   * @param cmdArgs
   * @throws FileNotFoundException 
   */
  public WriteKeyAcclPrc (WriteKeyAccl.CmdArgs cmdArgs ) {
    this.cmdArgs = cmdArgs;
    initKeyTranslate();
    OutTextPreparer gTxtKeyOutput = null;
    if(cmdArgs.file_gTxt !=null) {
      try {
        FileInputStream reader_gTxt = new FileInputStream(cmdArgs.file_gTxt);
        OutTextPreparer.readStreamTemplateCreatePreparer(reader_gTxt, null, this.idxScript, this, null);
        reader_gTxt.close();
        for(OutTextPreparer gTxt: this.idxScript.values()) {
          gTxt.parse(this.getClass(), null, this.idxScript);
        }
        gTxtKeyOutput = this.idxScript.get("KeyAcclText");
      } catch (Exception exc) {
        System.err.println(exc.getMessage());
      }
    }
    if(gTxtKeyOutput !=null) {
      this.gTxtKeyOutput = gTxtKeyOutput;
    } else {
      this.gTxtKeyOutput = gTxtKeyOutputDefault;
    }
  }

  
  /**Fills the {@link #idxKeyTranslateKeyNames}. Key names in XML to human readable key names. 
   * Its content is: <pre>
  private void initKeyTranslate () {
    this.idxKeyTranslate.put("KEY_BACKSPACE", "Bspc");
    this.idxKeyTranslate.put("KEY_DOWN", "Down");
    this.idxKeyTranslate.put("KEY_ESCAPE", "Esc");
    this.idxKeyTranslate.put("KEY_HOME", "Home");
    this.idxKeyTranslate.put("KEY_END", "End");
    this.idxKeyTranslate.put("KEY_INSERT", "Ins ");
    this.idxKeyTranslate.put("KEY_DELETE", "Del ");
    this.idxKeyTranslate.put("KEY_LEFT", "Left");
    this.idxKeyTranslate.put("KEY_PAGEDOWN", "PgDn");
    this.idxKeyTranslate.put("KEY_PAGEUP", "PgUp");
    this.idxKeyTranslate.put("KEY_RETURN", "ENTER");
    this.idxKeyTranslate.put("KEY_RIGHT", "Right");
    this.idxKeyTranslate.put("KEY_BRACKETLEFT", " [");
    this.idxKeyTranslate.put("KEY_BRACKETRIGHT", " ]");
    this.idxKeyTranslate.put("KEY_DIVIDE", " /");
    this.idxKeyTranslate.put("KEY_ADD", " +");
    this.idxKeyTranslate.put("KEY_SUBTRACT", " -");
    this.idxKeyTranslate.put("KEY_MULTIPLY", " *");
    this.idxKeyTranslate.put("KEY_NUMBERSIGN", " #");
    this.idxKeyTranslate.put("KEY_SPACE", "Space");
  }
   * </pre>*/
  private void initKeyTranslate () {
    this.idxKeyTranslateKeyNames.put("KEY_BACKSPACE", "Bspc");
    this.idxKeyTranslateKeyNames.put("KEY_DOWN", "Down");
    this.idxKeyTranslateKeyNames.put("KEY_ESCAPE", "Esc");
    this.idxKeyTranslateKeyNames.put("KEY_HOME", "Home");
    this.idxKeyTranslateKeyNames.put("KEY_END", "End");
    this.idxKeyTranslateKeyNames.put("KEY_INSERT", "Ins ");
    this.idxKeyTranslateKeyNames.put("KEY_DELETE", "Del ");
    this.idxKeyTranslateKeyNames.put("KEY_LEFT", "Left");
    this.idxKeyTranslateKeyNames.put("KEY_PAGEDOWN", "PgDn");
    this.idxKeyTranslateKeyNames.put("KEY_PAGEUP", "PgUp");
    this.idxKeyTranslateKeyNames.put("KEY_RETURN", "ENTER");
    this.idxKeyTranslateKeyNames.put("KEY_RIGHT", "Right");
    this.idxKeyTranslateKeyNames.put("KEY_BRACKETLEFT", " [");
    this.idxKeyTranslateKeyNames.put("KEY_BRACKETRIGHT", " ]");
    this.idxKeyTranslateKeyNames.put("KEY_DIVIDE", " /");
    this.idxKeyTranslateKeyNames.put("KEY_ADD", " +");
    this.idxKeyTranslateKeyNames.put("KEY_SUBTRACT", " -");
    this.idxKeyTranslateKeyNames.put("KEY_MULTIPLY", " *");
    this.idxKeyTranslateKeyNames.put("KEY_NUMBERSIGN", " #");
    this.idxKeyTranslateKeyNames.put("KEY_SPACE", "Space");
  }

  
  
  public void init(LogMessage log) {
    
    List<FileFunctions.FilePathnameExt> listFiles = new LinkedList<>();
    if(this.cmdArgs.dirLOffcPrg ==null || !this.cmdArgs.dirLOffcPrg.exists()) {
      log.writef("\nPROBLEM faulty Path for -LOffcPrg:%s", this.cmdArgs.dirLOffcPrg.getAbsolutePath());
    } else {
      try {
        FileFunctions.FilePathnameExt.parseWildcardZipPath(listFiles, this.cmdArgs.dirLOffcPrg, "share/registry/main.xcd", ':');
        FileFunctions.FilePathnameExt.parseWildcardZipPath(listFiles, this.cmdArgs.dirLOffcPrg, "share/registry/writer.xcd", ':');
        FileFunctions.FilePathnameExt.parseWildcardZipPath(listFiles, this.cmdArgs.dirLOffcPrg, "share/registry/draw.xcd", ':');
        FileFunctions.FilePathnameExt.parseWildcardZipPath(listFiles, this.cmdArgs.dirLOffcPrg, "share/registry/calc.xcd", ':');
        
        this.reg.readPrepare(listFiles, log);
      } catch (FileNotFoundException exc) {
        log.writeError("\nException parseWildcardZipPath ", exc);
      }
    }
    
//  ReadUiXcu_LOffc xcu = new ReadUiXcu_LOffc();
//
//  
//  
//  try {
//    FileFunctions.FilePathnameExt.parseWildcardZipPath(listFiles, null, "/home/D/LibreOfficeDev/libreoffice/officecfg/registry/data/org/openoffice/Office/UI/*.xcu", ':');
//    xcu.readPrepareXcuUi(listFiles, this.log);
//  } catch (FileNotFoundException exc) {
//    log.writeError("\nException parseWildcardZipPath ", exc);
//  }
//  
    
  }
  
  
  /**execution operation in the instantiated class, also called pure as is 
   * in {@link #exec(XmlDataLOffcKeyAccl, org.vishia.libOffc.cfgui.WriteKeyAccl.CmdArgs, LogMessage)}.
   * <ul>
   * <li>reads the 
   * </ul> 
   * @param data Data from XML read.
   * @return error number, 0.
   */
  public int exec(XmlDataLOffcKeyAccl data, File fIn, LogMessage log) {
    
    int nError = 0;
    for(XmlDataLOffcKeyAccl.Accel_item e: data.get_accel_acceleratorlist().get_accel_item()) {
      int nKeyVariant = 0;
      if(e.get_accel_shift() !=null) nKeyVariant |=1;      // builld an index with the modification keys.
      if(e.get_accel_mod1()  !=null) nKeyVariant |=2;
      if(e.get_accel_mod2()  !=null) nKeyVariant |=4;
      String[] sKeyVariant =    { "        ", "Shift-  ", "   Ctrl-", "Sh+Ctrl-", "        Alt-", "Shift + Alt-", "   Ctrl+Alt-", "Sh+Ctrl+Alt-"  };
      String[] sKeySortLOffc =  {"0",         "1",        "2",        "3",        "4",            "5",            "6",            "7" };
      String[] sKeySortLSh =    {"0",         "0",        "2",        "2",        "4",            "4",            "6",            "6" };     // sh variant after key without shift
      //---------------------------------------------------vv Key translate
      String sKeyRaw = e.get_accel_code();
      String sKeyOnly = this.idxKeyTranslateKeyNames.get(sKeyRaw);
      if(sKeyOnly == null) {
        sKeyOnly = sKeyRaw.startsWith("KEY_") ? sKeyRaw.substring(4) : sKeyRaw;
        if(sKeyOnly.length() < 2) { sKeyOnly = " " + sKeyOnly; }         // space before for distance if possible   
        if(sKeyOnly.length() < 4) { sKeyOnly += "    ".substring(sKeyOnly.length()); } // spaces after to ensure 4 chars
      }
      //---------------------------------------------------vv cmd .uno etc.
      String sCmd = e.get_xlink_href();                    // from XML
      ReadRegistryMain.UnoLabel unoLabel = this.reg.idxUnoLabel.get(sCmd);
      final String sLabel;             //------------------^^ try to translate the given uno command
      if(unoLabel == null) {                     //--------vv not translated:
        if(sCmd.startsWith(".uno:StyleApply?Style:string=")) { // special case '.uno:StyleApply' 
          int posSep = sCmd.indexOf('&', 29);
          if(sCmd.endsWith("&FamilyName:string=CharacterStyles")){
            sLabel = "characterStyleApply:" + sCmd.substring(29, posSep);
          }
          else if(sCmd.endsWith("&FamilyName:string=ParagraphStyles")){
            sLabel = "paragraphStyleApply:" + sCmd.substring(29, posSep);
          } else {
            sLabel = sCmd;                                 // proper text for '.uno:StyleApply'
          }
        } else {
          sLabel = sCmd;                                   // default: use uno command for output
        }
      } else {
        sLabel = unoLabel.textEn;                          // uno command found, use human readable text.
      }
      //------vvvv-----------------------------------------vv instance for the accelerator key translation:
      AcclKey acclKey = new AcclKey(sKeyOnly, sKeyVariant[nKeyVariant] + sKeyOnly, sLabel, sCmd);
      String sKeyAccl = sKeySortLOffc[nKeyVariant] + sKeyOnly;  // sorting keys for Map
      String sKeyAccl2 = sKeySortLSh[nKeyVariant] + sKeyOnly + "-" + sKeySortLOffc[nKeyVariant];
      this.idxKeyAccl.put(sKeyAccl, acclKey);
      this.idxKeyAcclShift.put(sKeyAccl2, acclKey);
      Debugutil.stop();  // look in debugger
    }
    Debugutil.stop();  // look in debugger

    try {
      outKeys(fIn, log);
    } catch (IOException e1) {
      // TODO Auto-generated catch block
      e1.printStackTrace();
    }
    
    return nError;
  }

  
  
  /**Outputs the accelerator key setting in textual human readable form
   * from the stored {@link #idxKeyAccl} or {@link #idxKeyAcclShift}.
   * It uses the prepared gTxt script in {@link #gTxtKeyOutput} which's content depends on the file 
   * given with {@link #cmdArgs} {@link WriteKeyAccl.CmdArgs#file_gTxt}
   * If this file is not given or faulty, then {@link #gTxtKeyOutputDefault} is used.  
   * @param fIn The input file may be used with its path/name without extension.
   *  The name of the output file is determined by {@link WriteKeyAccl.CmdArgs#fpOutText}
   * @param log
   * @throws IOException
   */
  void outKeys (File fIn, LogMessage log) throws IOException {
                                       //------vvvv--------vv The used output file for accelerator key human text. */
    File fOut = this.cmdArgs.fpOutText.newFileFromDirName(fIn);
    log.writef("\nwrites accelerator key text: %s in %s/ ...", fOut.getName(), fOut.getParentFile().getAbsolutePath() );
    FileWriter wOut = new FileWriter(fOut);                // FileWriter for OutTextPreparer
    OutTextPreparer.DataTextPreparer gdata = this.gTxtKeyOutput.createArgumentDataObj();
    //                                 
    gdata.setArgument("thiz", this);
    this.gTxtKeyOutput.exec(wOut, gdata);
    wOut.close();
    log.writeInfo(" done\n" );
  }
  

}
