package org.vishia.libOffc.cfgui;

import java.io.IOException;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.vishia.msgDispatch.LogMessage;
import org.vishia.util.Debugutil;
import org.vishia.util.FileFunctions;
import org.vishia.xmlReader.XmlCfg;
import org.vishia.xmlReader.XmlJzReader;

/**This class reads some given 'share/registry/*.xcd' files in the LibreOffice installation directory.
 * <ul>
 * <li>The configuration for {@link XmlJzReader}, {@link XmlCfg} was created with {@link org.vishia.xmlReader.XmlJzCfgAnalyzer}
 *   and is stored in {@link registryMain.xml.cfg} beside this class in jar.
 * <li>The {@link XmlDataRegMain} with its super class {@link XmlDataRegMain_Base} is used to store the data from XML in a semantic form.
 * <li> {@link XmlDataRegMain_Zbnf} is used as facade class to store the data.
 * <li>After XML parsing the data are prepared and stored in {@link #idxUnoLabel} with the instance of {@link UnoLabel}
 * </ul> 
 */
public class ReadRegistryMain {
  
  public static class UnoLabel {
    public final String unoCmd;
    public final String textEn;
    public final String component;
    public final String modules;
    
    public UnoLabel( String unoCmd, String textEn, String component, String modules ) {
      this.unoCmd = unoCmd;
      this.textEn = textEn;
      this.component = component;
      this.modules = modules;
    }
    
    @Override public String toString() {
      return this.unoCmd + "=\"" + this.textEn + "\" : " + this.component + ":" + this.modules; 
    }
    
  }
  
  
  /**Index between the uno command as key and the human readable text,
   * which comes from a menu entry etc. in the xcd files, 
   * where this uno command is used.
   * It is not an official translation or explanation of uno commands,
   * it is gathered by the way from other command settings stored in the xcd files,
   * and though helpfully.
   */
  Map<String, UnoLabel> idxUnoLabel = new TreeMap<>(); 
  
  /**Index only internally to explore in debug, which uno commands are assigned
   * to more as one text.*/
  Map<String, Map<String, UnoLabel>> idxCmpnUno = new TreeMap<>(); 
  
  
  /**empty ctor
   * 
   */
  public ReadRegistryMain () {
  }
  
  
  /**The operation to read all given xcd an prepare the {@link #idxUnoLabel}.
   * <ul>
   * <li>calls {@link #readRegistryMain(List, LogMessage)} as XML reader wrapper, that uses the {@link XmlJzReader} concept.
   * <li>calls {@link #prepareRegistryNode(XmlDataRegMain)} which creates content of {@link #idxCmpnUno}.
   * </ul>
   * @param listFiles all files to read. 
   *   This list is to be filled with {@link FileFunctions.FilePathnameExt#parseWildcardZipPath(java.util.Collection, java.io.File, String, char)}.
   *   It means it can be filled with several files given also with wild cards, 
   *   and if necessary also with a given path in the file to the inner XML, if the file is a zip file. 
   *   This capability is not used here, but the operation is also proper for XML in zip (can be used as template for the {@link XmlJzReader}).
   *   The file to read is each {@link FileFunctions.FilePathnameExt#file}.
   *   The {@link FileFunctions.FilePathnameExt#sNamePath} is null for this approach, can be set as path/to.xml in the zipfile
   * @param log for logging.
   */
  public void readPrepare (List<FileFunctions.FilePathnameExt> listFiles, LogMessage log) {
    XmlDataRegMain data = readRegistryMain(listFiles, log);
    prepareRegistryNode(data);
    Debugutil.stop();
  }


  /**Read and prepare the *.xcd XML files.
   * @param listFiles all files to read. See {@link #readPrepare(List, LogMessage)}.
   * @param log for log
   */
  private XmlDataRegMain readRegistryMain (List<FileFunctions.FilePathnameExt> listFiles, LogMessage log) {
    XmlCfg xmlCfg = new XmlCfg();
    XmlDataRegMain_Zbnf data = new XmlDataRegMain_Zbnf();
    try {
      xmlCfg.readFromJar(this.getClass(), "registryMain.xml.cfg", log);
      XmlJzReader xmlReader = new XmlJzReader(log);
      xmlReader.setCfg(xmlCfg);
      for(FileFunctions.FilePathnameExt fIn1: listFiles) {
        String sInName = fIn1.file.getName();
        try {
          log.writef("\nreads XML: %s", fIn1.file.getAbsolutePath());
          if(fIn1.sNamePath !=null) {
            xmlReader.readZipXml(fIn1.file, fIn1.sNamePath, data);
          } else {
            xmlReader.readXml(fIn1.file, data, null);
          }
        } catch(Exception exc) {
          log.writeError("\nEXCEPTION parse : " + sInName + ": ", exc);
        }
      }
    } catch (IllegalCharsetNameException | UnsupportedCharsetException | IOException exc) {
      log.writeError("\nEXCEPTION xmlCfg : ", exc);
    }
    return data.dataroot;
  }
  
  
  /**Iterates through the XML data, recursively,
   * <ul>
   * <li>for all {@link XmlDataRegMain_Base#oor_data}
   * <ul><li>for all {@link XmlDataRegMain.Oor_data#oor_component_data}
   *     <ul><li>for all {@link XmlDataRegMain.XmlDataRegMain_Base.Oor_component_data_Base#node}
   *         <li>calls {@link #prepareNode(org.vishia.libOffc.cfgui.XmlDataRegMain.Node, String, String)}
   *     </ul>    
   * </ul></ul>
   * @param data The parsed XML data.
   */
  void prepareRegistryNode ( XmlDataRegMain data ) {
    for(XmlDataRegMain.Oor_data xcmpn:  data.oor_data){
      for(XmlDataRegMain.Oor_component_data xcmpnd : xcmpn.get_oor_component_data()) {
        String nameCmpn = xcmpnd.oor_name;
        Iterable<XmlDataRegMain.Node> iterNode = xcmpnd.get_node();
        if(iterNode !=null) for(XmlDataRegMain.Node xnode : iterNode){
          prepareNode(xnode, "", nameCmpn);
        }
      }
      
      Debugutil.stop();
//      String sCmpn = xcmpn.oor_name;
//      for(XmlDataRegMain.Node xnode: xcmpn.node) {
//        prepareNode(xnode, "", sCmpn);
//      }
    }
    
    
    
    Debugutil.stop();
    
  }
  
  
  /**Prepare all &lt;node>...&lt;/node> in XML:
   * <ul><li>if {@link XmlDataRegMain_Base.Node_Base#oor_name} starts with ".uno" then it is a uno node.
   *   Then calls {@link #prepareNodeUno(org.vishia.libOffc.cfgui.XmlDataRegMain.Node, String, String)}.
   * <li>if sub nodes exists, calls this recursively. The sub nodes may contain uno nodes.
   * </ul>
   * @param xnode
   * @param sNodeNamesParent
   * @param sCmpn
   */
  void prepareNode (XmlDataRegMain.Node xnode, String sNodeNamesParent, String sCmpn) {
    if(xnode.oor_name.startsWith(".uno")) {
      prepareNodeUno(xnode, sNodeNamesParent, sCmpn);
    }
    if(xnode.oor_op !=null && xnode.oor_op.equals("replace")) {
      //prepareNodeReplace(xnode, sNodenames, sCmpn);
    } else {
      Debugutil.stop();
    }
    String sNodenames = sNodeNamesParent + "/" + xnode.oor_name;
    if(xnode.node !=null) for(XmlDataRegMain.Node xnodeSub: xnode.node) {
      prepareNode(xnodeSub, sNodenames, sCmpn);
    }
    
  }

  
  /**Called if a uno-node is found: {@link XmlDataRegMain_Base.Node_Base#oor_name} starts with ".uno".
   * it sets one element of {@link #idxUnoLabel} and also add in {@link #idxCmpnUno}.
   * @param xnode The uno node from XML
   * @param sNodeNames String build with all parent node names.
   * @param sCmpn String from the component as parent. 
   */
  void prepareNodeUno (XmlDataRegMain.Node xnode, String sNodeNames, String sCmpn) {
    String sLabelUno = null;
    for(XmlDataRegMain.Prop xprop: xnode.prop) {
      if(xprop.oor_name.equals("Label")) {
        for(XmlDataRegMain.Value xvalue: xprop.value) {
          sLabelUno = xvalue.get__textContent();
        }
      }
    }
    if(sLabelUno !=null) {
      UnoLabel unoLabel = new UnoLabel(xnode.oor_name, sLabelUno, sCmpn, sNodeNames);
      this.idxUnoLabel.put(xnode.oor_name, unoLabel);
      String keyCmpn = sCmpn + sNodeNames;
      Map<String, UnoLabel> unoLabelCmpn = this.idxCmpnUno.get(keyCmpn);
      if(unoLabelCmpn == null) {
        unoLabelCmpn = new TreeMap<>();
        this.idxCmpnUno.put(keyCmpn, unoLabelCmpn);
      }
      unoLabelCmpn.put(xnode.oor_name, unoLabel);
    }
  }
  
  
  
}
